/*-
 * Copyright (c) 2013 Yahoo!, Inc.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *
 * $FreeBSD: releng/11.1/usr.sbin/mfiutil/mfi_properties.c 271882 2014-09-19 23:38:44Z smh $
 */

#include <sys/param.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "mfiutil.h"

MFI_TABLE(top, ctrlprop);

static void convert_little_endian_values(struct mfi_ctrl_props *props) {
	CONVLE16(&props->seq_num);
	CONVLE16(&props->pred_fail_poll_interval);
	CONVLE16(&props->intr_throttle_cnt);
	CONVLE16(&props->intr_throttle_timeout);
	CONVLE16(&props->ecc_bucket_leak_rate);
	CONVLE16(&props->spinDownTime);
}

static int
mfi_ctrl_get_properties(int fd, struct mfi_ctrl_props *info)
{
	if(mfi_dcmd_command(fd, MFI_DCMD_CTRL_GET_PROPS, info,
	    sizeof(struct mfi_ctrl_props), NULL, 0, NULL) < 0) {
		return -1;
	}
	convert_little_endian_values(info);
	return 0;
}

static int
mfi_ctrl_set_properties(int fd, struct mfi_ctrl_props *info)
{
	convert_little_endian_values(info);
	return (mfi_dcmd_command(fd, MFI_DCMD_CTRL_SET_PROPS, info,
	    sizeof(struct mfi_ctrl_props), NULL, 0, NULL));
}

/*
 * aquite the controller properties data structure modify the 
 * rebuild rate if requested and then retun
 */
static int
mfi_ctrl_rebuild_rate(int ac, char **av)
{
	int error, fd;
	struct mfi_ctrl_props ctrl_props;

	if (ac > 2) {
		warnx("Extra operands");
		return EINVAL;
	}

	fd = mfi_open(mfi_unit, O_RDWR);
	if (fd < 0) {
		error = errno;
		warn("mfi_open");
		return (error);
	}

	error = mfi_ctrl_get_properties(fd, &ctrl_props);
	if ( error < 0) {
		error = errno;
		warn("Failed to get controller properties");
		close(fd);
		return (error);
	}
	/*
	 * User requested a change to the rebuild rate
	 */
	if (ac > 1) {
		ctrl_props.rebuild_rate = atoi(av[ac - 1]);
		error = mfi_ctrl_set_properties(fd, &ctrl_props);
		if ( error < 0) {
			error = errno;
			warn("Failed to set controller properties");
			close(fd);
			return (error);
		}

		error = mfi_ctrl_get_properties(fd, &ctrl_props);
		if ( error < 0) {
			error = errno;
			warn("Failed to get controller properties");
			close(fd);
			return (error);
		}
	}
	printf ("controller rebuild rate: %%%u \n",
		ctrl_props.rebuild_rate);
	return (0);
}
MFI_COMMAND(ctrlprop, rebuild, mfi_ctrl_rebuild_rate);

static int
mfi_ctrl_alarm_enable(int ac, char **av)
{
	int error, fd;
	struct mfi_ctrl_props ctrl_props;

	if (ac > 2) {
		warnx("Extra operands");
		return EINVAL;
	}

	fd = mfi_open(mfi_unit, O_RDWR);
	if (fd < 0) {
		error = errno;
		warn("mfi_open");
		return (error);
	}

	error = mfi_ctrl_get_properties(fd, &ctrl_props);
	if ( error < 0) {
		error = errno;
		warn("Failed to get controller properties");
		close(fd);
		return (error);
	}
	printf ("controller alarm was %s\n",
		(ctrl_props.alarm_enable ? "enabled" : "disabled"));
	if (ac > 1) {
		if(strcasecmp(av[1], "off") == 0) ctrl_props.alarm_enable = 0;
		else if(strcasecmp(av[1], "on") == 0) ctrl_props.alarm_enable = 1;
		else if((av[1][0] == '0' || av[1][0] == '1') && !av[1][1]) ctrl_props.alarm_enable = av[1][0] - '0';
		else {
			warnx("Incorrect property value");
			close(fd);
			return EINVAL;
		}
		error = mfi_ctrl_set_properties(fd, &ctrl_props);
		if ( error < 0) {
			error = errno;
			warn("Failed to set controller properties");
			close(fd);
			return (error);
		}

		error = mfi_ctrl_get_properties(fd, &ctrl_props);
		if ( error < 0) {
			error = errno;
			warn("Failed to get controller properties");
			close(fd);
			return (error);
		}
		printf("set to %s\n", ctrl_props.alarm_enable ? "enabled" : "disabled");
	}
	return (0);
}

MFI_COMMAND(ctrlprop, alarm, mfi_ctrl_alarm_enable);
